#!/bin/sh
. /lib/netifd/mac80211.sh

append DRIVERS "mac80211"

lookup_phy() {
	[ -n "$phy" ] && {
		[ -d /sys/class/ieee80211/$phy ] && return
	}

	local devpath
	config_get devpath "$device" path
	[ -n "$devpath" ] && {
		phy="$(mac80211_path_to_phy "$devpath")"
		[ -n "$phy" ] && return
	}

	local macaddr="$(config_get "$device" macaddr | tr 'A-Z' 'a-z')"
	[ -n "$macaddr" ] && {
		for _phy in /sys/class/ieee80211/*; do
			[ -e "$_phy" ] || continue

			[ "$macaddr" = "$(cat ${_phy}/macaddress)" ] || continue
			phy="${_phy##*/}"
			return
		done
	}
	phy=
	return
}

find_mac80211_phy() {
	local device="$1"

	config_get phy "$device" phy
	lookup_phy
	[ -n "$phy" -a -d "/sys/class/ieee80211/$phy" ] || {
		echo "PHY for wifi device $1 not found"
		return 1
	}
	config_set "$device" phy "$phy"

	config_get macaddr "$device" macaddr
	[ -z "$macaddr" ] && {
		config_set "$device" macaddr "$(cat /sys/class/ieee80211/${phy}/macaddress)"
	}

	return 0
}

check_mac80211_device() {
	config_get phy "$1" phy
	[ -z "$phy" ] && {
		find_mac80211_phy "$1" >/dev/null || return 0
		config_get phy "$1" phy
	}
	[ "$phy" = "$dev" ] && found=1
}

detect_mac80211() {
	devidx=0
	config_load wireless
	while :; do
		config_get type "radio$devidx" type
		[ -n "$type" ] || break
		devidx=$(($devidx + 1))
	done
	
	NASID=`head /dev/urandom | tr -dc "0123456789" | head -c11`

	for _dev in /sys/class/ieee80211/*; do
		[ -e "$_dev" ] || continue

		dev="${_dev##*/}"

		found=0
		config_foreach check_mac80211_device wifi-device
		[ "$found" -gt 0 ] && continue

		mode_band="g"
		channel="11"
		htmode=""
		ht_capab=""

		iw phy "$dev" info | grep -q 'Capabilities:' && htmode=HT20

		iw phy "$dev" info | grep -q '\* 5... MHz \[' && {
			mode_band="a"
			channel=$(iw phy "$dev" info | grep '\* 5... MHz \[' | grep '(disabled)' -v -m 1 | sed 's/[^[]*\[\|\].*//g')
			iw phy "$dev" info | grep -q 'VHT Capabilities' && htmode="VHT80"
		}

		iw phy "$dev" info | grep -q '\* 5.... MHz \[' && {
			mode_band="ad"
			channel=$(iw phy "$dev" info | grep '\* 5.... MHz \[' | grep '(disabled)' -v -m 1 | sed 's/[^[]*\[\|\|\].*//g')
			iw phy "$dev" info | grep -q 'Capabilities:' && htmode="HT20"
		}

		[ -n "$htmode" ] && ht_capab="set wireless.radio${devidx}.htmode=$htmode"

		path="$(mac80211_phy_to_path "$dev")"
		if [ -n "$path" ]; then
			dev_id="set wireless.radio${devidx}.path='$path'"
		else
			dev_id="set wireless.radio${devidx}.macaddr=$(cat /sys/class/ieee80211/${dev}/macaddress)"
		fi

		uci -q batch <<-EOF
			set wireless.radio${devidx}=wifi-device
			set wireless.radio${devidx}.type=mac80211
			set wireless.radio${devidx}.channel=${channel}
			set wireless.radio${devidx}.hwmode=11${mode_band}
			set wireless.radio${devidx}.disabled='0'
			${dev_id}
			${ht_capab}
					
			
			set wireless.default_radio${devidx}=wifi-iface
			set wireless.default_radio${devidx}.device=radio${devidx}
			set wireless.default_radio${devidx}.network=lan
			set wireless.default_radio${devidx}.isolate='1'
			set wireless.default_radio${devidx}.mode=ap
			set wireless.default_radio${devidx}.ssid='Guest WiFi'
			set wireless.default_radio${devidx}.encryption=none
			set wireless.default_radio${devidx}.disabled='0'
			set wireless.default_radio${devidx}.ieee80211r='1'
			set wireless.default_radio${devidx}.ieee80211k='1'
			set wireless.default_radio${devidx}.ieee80211v='1'
			set wireless.default_radio${devidx}.nasid=${devidx}${NASID}
			set wireless.default_radio${devidx}.mobility_domain='d0ba'
			set wireless.default_radio${devidx}.ft_psk_generate_local='1'

			set wireless.guest${devidx}=wifi-iface
			set wireless.guest${devidx}.device=radio${devidx}
			set wireless.guest${devidx}.mode=ap
			set wireless.guest${devidx}.network=guest${devidx}
			set wireless.guest${devidx}.isolate='1'
			set wireless.guest${devidx}.ssid='Staff WiFi'
			set wireless.guest${devidx}.encryption=psk2
			set wireless.guest${devidx}.key='starthotspot'
			set wireless.guest${devidx}.disabled='1'			
			
EOF
		uci -q commit wireless

		devidx=$(($devidx + 1))
	done
}
